from __future__ import absolute_import, print_function, unicode_literals
import Live
from ableton.v2.control_surface import Component
from ableton.v2.base import liveobj_valid


class SetlistManager(Component):
    def __init__(self, parent, *a, **k):
        super(SetlistManager, self).__init__(*a, **k)
        self.parent_instance = parent
        self.song_pool = {}
        self.setlist = []
        self.setlist_to_store = {}
        self.setlist_to_recall = {}
        self.setlist_index = None
    
    def recall_setlist(self):
        try:
            # Retrieve the stored setlist from Ableton
            self.setlist_to_recall = self.parent_instance.song.get_data("stored_setlist", None)
            self.parent_instance._log("Retrieved stored setlist: {}".format(self.setlist_to_recall))

            if not isinstance(self.setlist_to_recall, dict):
                self.parent_instance._log("Stored setlist is not in the expected dictionary format.")
                return

            if not self.setlist_to_recall:
                self.parent_instance._log("No stored setlist found.")
                return

            # Prepare for O(1) lookups by reversing song_cues: map names and times to cue IDs
            cue_points = {
                (cue_data["name"], cue_data["time"], cue_data["length"]): cue_id
                for cue_id, cue_data in self.parent_instance.song_cues.items()
            }

            # Match stored songs with cues in the current session
            for stored_song_id, stored_song in self.setlist_to_recall.items():
                key = (stored_song["name"], stored_song["time"], stored_song["length"])
                if key in cue_points:
                    cue_id = cue_points[key]
                    self.setlist.append({"id": cue_id, "name": stored_song["name"], "length": stored_song["length"]})

        except Exception as e:
            self.parent_instance._log("Error recalling setlist: {}".format(e))
            return

        self.parent_instance._log("Recalled Setlist: {}".format(self.setlist))
        self.setlist_index = 0  # Initialize to the first song



        
#    @property
    def get_setlist_index(self):
        """Getter for the setlist index."""
        return self._setlist_index

#    @setlist_index.setter
    def set_setlist_index(self, value):
        """Setter for the setlist index, triggers a function."""
        if value is None:
            self._setlist_index = None
            self.parent_instance.current_song_cue = None
        elif 0 <= value < len(self.setlist):
            self._setlist_index = value
            current_song_id = self.setlist[self._setlist_index]["id"]
            self.parent_instance.selected_cue(current_song_id)
            self.parent_instance.socket_manager.server.send_setlist_index_update(self._setlist_index)
        else:
            self.parent_instance._log("Invalid index {}, skipping cue selection.".format(value))
            
    setlist_index = property(get_setlist_index, set_setlist_index)


    def refresh_song_pool(self):
        updated_song_pool = {key: {"name": value["name"], "length": value["length"]} for key, value in self.parent_instance.song_cues.items()}
        self.song_pool = updated_song_pool

        if self.setlist:
            self.setlist = [
                {
                    "id": song["id"],
                    "name": song["name"],
                    "length": updated_song_pool[song["id"]]["length"]
                }
                for song in self.setlist
                if song["id"] in updated_song_pool
            ]
            if self.setlist:
                # Update stored data if setlist is not empty
                self.setlist_to_store = {
                    song["id"]: {
                        "name": updated_song_pool[song["id"]]["name"],
                        "time": self.parent_instance.song_cues[song["id"]]["time"],
                        "length": updated_song_pool[song["id"]]["length"]
                    }
                    for song in self.setlist
                }
                
                cue = self.parent_instance.current_song_cue
                if cue is not None and not liveobj_valid(cue):
                    if self.setlist_index > len(self.setlist) - 1:
                        self.setlist_index = len(self.setlist) - 1
                    else:
                        current_index = self.setlist_index
                        self.setlist_index = current_index
                    
            else:
                # Clear stored data if setlist is now empty
                self.setlist_to_store = {}

            # Persist changes
            self.parent_instance.song.set_data("stored_setlist", self.setlist_to_store)
            self.parent_instance._log("Refreshed Setlist: {}".format(self.setlist))

        # Mark the song pool as refreshed
        self.parent_instance.socket_manager.server.song_pool_refreshed = True


        
    def update_setlist(self, songs, client=None):
        if songs == []:
            self.parent_instance.song.set_data("stored_setlist", [])
            self.setlist = []
            self.setlist_index = None
            self.parent_instance.socket_manager.server.song_pool_refreshed = True
            self.parent_instance.socket_manager.server.excluded_client = client
            return
        self.setlist = songs
        self.setlist_to_store = {
            song["id"]: {
                "name": song["name"],
                "time": self.parent_instance.song_cues[song["id"]]["time"],
                "length": self.parent_instance.song_cues[song["id"]]["length"]
            }
            for song in self.setlist
        }
        self.parent_instance._log("Storing setlist: {}".format(self.setlist_to_store))
        self.parent_instance.song.set_data("stored_setlist", self.setlist_to_store)
        self.parent_instance._log("Updated Setlist: {}".format(self.setlist))
        self.parent_instance.socket_manager.server.song_pool_refreshed = True
        self.parent_instance.socket_manager.server.excluded_client = client
        
